<?php
/**
 * Plugin Name: Growffinity CRM for WooCommerce
 * Plugin URI: https://growffinity.com
 * Description: Connect your WooCommerce store to Growffinity CRM. Sync customers, orders, and manage your business from one powerful dashboard.
 * Version: 1.0.3
 * Author: Growffinity
 * Author URI: https://growffinity.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: growffinity-crm
 * Domain Path: /languages
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * WC requires at least: 5.0
 * WC tested up to: 9.0
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('GROWFFINITY_VERSION', '1.0.3');
define('GROWFFINITY_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('GROWFFINITY_PLUGIN_URL', plugin_dir_url(__FILE__));
define('GROWFFINITY_API_URL', 'https://woocommerce-crm-backend.onrender.com/api');

/**
 * Main Growffinity CRM Class
 */
class Growffinity_CRM {

    /**
     * Instance of this class
     */
    private static $instance = null;

    /**
     * Get instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->init_hooks();
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Check if WooCommerce is active
        add_action('admin_init', array($this, 'check_woocommerce'));

        // Admin menu
        add_action('admin_menu', array($this, 'add_admin_menu'));

        // Admin scripts and styles
        add_action('admin_enqueue_scripts', array($this, 'admin_scripts'));

        // AJAX handlers
        add_action('wp_ajax_growffinity_connect_store', array($this, 'ajax_connect_store'));
        add_action('wp_ajax_growffinity_disconnect_store', array($this, 'ajax_disconnect_store'));
        add_action('wp_ajax_growffinity_sync_now', array($this, 'ajax_sync_now'));

        // Automatic sync hooks
        add_action('woocommerce_new_order', array($this, 'sync_order'), 10, 1);
        add_action('woocommerce_update_order', array($this, 'sync_order'), 10, 1);
        add_action('woocommerce_created_customer', array($this, 'sync_customer'), 10, 1);
        add_action('woocommerce_update_customer', array($this, 'sync_customer'), 10, 1);

        // Activation/deactivation hooks
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }

    /**
     * Check if WooCommerce is active
     */
    public function check_woocommerce() {
        if (!class_exists('WooCommerce')) {
            add_action('admin_notices', array($this, 'woocommerce_missing_notice'));
            deactivate_plugins(plugin_basename(__FILE__));
        }
    }

    /**
     * WooCommerce missing notice
     */
    public function woocommerce_missing_notice() {
        ?>
        <div class="error">
            <p><?php esc_html_e('Growffinity CRM requires WooCommerce to be installed and active.', 'growffinity-crm'); ?></p>
        </div>
        <?php
    }

    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_menu_page(
            __('Growffinity CRM', 'growffinity-crm'),
            __('Growffinity CRM', 'growffinity-crm'),
            'manage_woocommerce',
            'growffinity-crm',
            array($this, 'admin_page'),
            'dashicons-chart-area',
            56
        );
    }

    /**
     * Admin scripts and styles
     */
    public function admin_scripts($hook) {
        if ('toplevel_page_growffinity-crm' !== $hook) {
            return;
        }

        wp_enqueue_style('growffinity-admin', GROWFFINITY_PLUGIN_URL . 'assets/css/admin.css', array(), GROWFFINITY_VERSION);
        wp_enqueue_script('growffinity-admin', GROWFFINITY_PLUGIN_URL . 'assets/js/admin.js', array('jquery'), GROWFFINITY_VERSION, true);

        wp_localize_script('growffinity-admin', 'growffinityData', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('growffinity_nonce'),
            'isConnected' => $this->is_connected(),
            'apiKey' => get_option('growffinity_api_key', ''),
        ));
    }

    /**
     * Admin page
     */
    public function admin_page() {
        require_once GROWFFINITY_PLUGIN_DIR . 'includes/admin-page.php';
    }

    /**
     * Check if store is connected
     */
    public function is_connected() {
        $api_key = get_option('growffinity_api_key', '');
        $store_id = get_option('growffinity_store_id', '');
        return !empty($api_key) && !empty($store_id);
    }

    /**
     * AJAX: Connect store
     */
    public function ajax_connect_store() {
        check_ajax_referer('growffinity_nonce', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => 'Permission denied'));
        }

        $email = sanitize_email($_POST['email']);
        $password = sanitize_text_field($_POST['password']);

        if (empty($email) || empty($password)) {
            wp_send_json_error(array('message' => 'Email and password are required'));
        }

        // Authenticate with Growffinity API
        $response = $this->authenticate_with_api($email, $password);

        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => $response->get_error_message()));
        }

        // Save API key and store ID
        update_option('growffinity_api_key', $response['token']);
        update_option('growffinity_user_id', $response['userId']);
        update_option('growffinity_user_email', $email);

        // Create or connect store
        $store_response = $this->connect_store_to_api($response['token']);

        if (is_wp_error($store_response)) {
            wp_send_json_error(array('message' => $store_response->get_error_message()));
        }

        update_option('growffinity_store_id', $store_response['storeId']);
        update_option('growffinity_connected_at', current_time('mysql'));

        wp_send_json_success(array(
            'message' => 'Successfully connected to Growffinity!',
            'storeId' => $store_response['storeId']
        ));
    }

    /**
     * AJAX: Disconnect store
     */
    public function ajax_disconnect_store() {
        check_ajax_referer('growffinity_nonce', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => 'Permission denied'));
        }

        delete_option('growffinity_api_key');
        delete_option('growffinity_store_id');
        delete_option('growffinity_user_id');
        delete_option('growffinity_user_email');
        delete_option('growffinity_connected_at');

        wp_send_json_success(array('message' => 'Successfully disconnected'));
    }

    /**
     * AJAX: Sync now
     */
    public function ajax_sync_now() {
        check_ajax_referer('growffinity_nonce', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => 'Permission denied'));
        }

        if (!$this->is_connected()) {
            wp_send_json_error(array('message' => 'Store not connected'));
        }

        $result = $this->perform_full_sync();

        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
        }

        $customers_synced = isset($result['customers']) ? $result['customers'] : 0;
        $orders_synced = isset($result['orders']) ? $result['orders'] : 0;

        wp_send_json_success(array(
            'message' => sprintf('Sync completed! %d customers and %d orders synced to Growffinity.', $customers_synced, $orders_synced),
            'stats' => $result
        ));
    }

    /**
     * Authenticate with Growffinity API
     */
    private function authenticate_with_api($email, $password) {
        $response = wp_remote_post(GROWFFINITY_API_URL . '/auth/login', array(
            'body' => json_encode(array(
                'email' => $email,
                'password' => $password
            )),
            'headers' => array(
                'Content-Type' => 'application/json'
            ),
            'timeout' => 15
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        $status_code = wp_remote_retrieve_response_code($response);

        if ($status_code !== 200) {
            return new WP_Error('auth_failed', $body['error'] ?? 'Authentication failed');
        }

        return $body;
    }

    /**
     * Connect store to API
     */
    private function connect_store_to_api($token) {
        $store_data = array(
            'name' => get_bloginfo('name'),
            'url' => get_site_url(),
            'woocommerce_version' => WC()->version,
            'wordpress_version' => get_bloginfo('version'),
        );

        $response = wp_remote_post(GROWFFINITY_API_URL . '/stores', array(
            'body' => json_encode($store_data),
            'headers' => array(
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $token
            ),
            'timeout' => 15
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        $status_code = wp_remote_retrieve_response_code($response);

        if ($status_code !== 200 && $status_code !== 201) {
            return new WP_Error('connection_failed', $body['error'] ?? 'Failed to connect store');
        }

        return array('storeId' => $body['id']);
    }

    /**
     * Perform full sync
     */
    private function perform_full_sync() {
        $api_key = get_option('growffinity_api_key');
        $store_id = get_option('growffinity_store_id');

        if (!$api_key || !$store_id) {
            return new WP_Error('not_connected', 'Store not connected');
        }

        $results = array(
            'customers' => 0,
            'orders' => 0
        );

        // Sync customers
        $customer_result = $this->sync_customers_bulk($api_key, $store_id);
        if (!is_wp_error($customer_result)) {
            $results['customers'] = $customer_result;
        }

        // Sync orders
        $order_result = $this->sync_orders_bulk($api_key, $store_id);
        if (!is_wp_error($order_result)) {
            $results['orders'] = $order_result;
        }

        return $results;
    }

    /**
     * Sync all customers in bulk
     */
    private function sync_customers_bulk($api_key, $store_id) {
        if (!class_exists('WC_Customer')) {
            return new WP_Error('wc_not_active', 'WooCommerce is not active');
        }

        // Get all WooCommerce customers (limit to 500 for performance)
        $customers = get_users(array(
            'role' => 'customer',
            'number' => 500
        ));

        error_log('Growffinity: Found ' . count($customers) . ' customers to sync');

        $customer_data = array();

        foreach ($customers as $user) {
            try {
                $customer = new WC_Customer($user->ID);

                $customer_data[] = array(
                    'id' => $user->ID,
                    'first_name' => $customer->get_first_name(),
                    'last_name' => $customer->get_last_name(),
                    'email' => $customer->get_email(),
                    'billing' => array(
                        'phone' => $customer->get_billing_phone()
                    ),
                    'total_spent' => $customer->get_total_spent(),
                    'orders_count' => $customer->get_order_count()
                );
            } catch (Exception $e) {
                error_log('Growffinity: Error processing customer ' . $user->ID . ': ' . $e->getMessage());
            }
        }

        error_log('Growffinity: Prepared ' . count($customer_data) . ' customers for API');

        if (empty($customer_data)) {
            return 0;
        }

        // Send to Growffinity API
        $response = wp_remote_post(GROWFFINITY_API_URL . '/sync/customers/' . $store_id . '/bulk', array(
            'body' => json_encode(array('customers' => $customer_data)),
            'headers' => array(
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $api_key
            ),
            'timeout' => 60
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        $status_code = wp_remote_retrieve_response_code($response);

        if ($status_code !== 200) {
            return new WP_Error('sync_failed', $body['error'] ?? 'Failed to sync customers');
        }

        return $body['inserted'] + $body['updated'];
    }

    /**
     * Sync all orders in bulk
     */
    private function sync_orders_bulk($api_key, $store_id) {
        if (!function_exists('wc_get_orders')) {
            return new WP_Error('wc_not_active', 'WooCommerce is not active');
        }

        // Get recent WooCommerce orders (limit to 500 for performance)
        $orders = wc_get_orders(array(
            'limit' => 500,
            'orderby' => 'date',
            'order' => 'DESC',
            'type' => 'shop_order' // Only get regular orders, not refunds
        ));

        error_log('Growffinity: Found ' . count($orders) . ' orders to sync');

        $order_data = array();

        foreach ($orders as $order) {
            try {
                // Skip if not a WC_Order object (safety check)
                if (!is_a($order, 'WC_Order')) {
                    error_log('Growffinity: Skipping non-order object: ' . get_class($order));
                    continue;
                }

                $order_data[] = array(
                    'id' => $order->get_id(),
                    'number' => $order->get_order_number(),
                    'customer_id' => $order->get_customer_id(),
                    'total' => $order->get_total(),
                    'status' => $order->get_status(),
                    'date_created' => $order->get_date_created()->date('Y-m-d H:i:s')
                );
            } catch (Exception $e) {
                error_log('Growffinity: Error processing order ' . $order->get_id() . ': ' . $e->getMessage());
            }
        }

        error_log('Growffinity: Prepared ' . count($order_data) . ' orders for API');

        if (empty($order_data)) {
            return 0;
        }

        // Send to Growffinity API
        $response = wp_remote_post(GROWFFINITY_API_URL . '/sync/orders/' . $store_id . '/bulk', array(
            'body' => json_encode(array('orders' => $order_data)),
            'headers' => array(
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $api_key
            ),
            'timeout' => 60
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        $status_code = wp_remote_retrieve_response_code($response);

        if ($status_code !== 200) {
            return new WP_Error('sync_failed', $body['error'] ?? 'Failed to sync orders');
        }

        return $body['inserted'] + $body['updated'];
    }

    /**
     * Sync single order
     */
    public function sync_order($order_id) {
        if (!$this->is_connected()) {
            return;
        }

        // Implementation will sync order to API
    }

    /**
     * Sync single customer
     */
    public function sync_customer($customer_id) {
        if (!$this->is_connected()) {
            return;
        }

        // Implementation will sync customer to API
    }

    /**
     * Plugin activation
     */
    public function activate() {
        // Set default options
        add_option('growffinity_auto_sync', 'yes');
        add_option('growffinity_sync_frequency', 'realtime');
    }

    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Cleanup if needed
    }
}

// Initialize the plugin
function growffinity_crm_init() {
    return Growffinity_CRM::get_instance();
}

// Start the plugin
add_action('plugins_loaded', 'growffinity_crm_init');
